//---------------------------------------------------------------------------
#include <vcl.h>
#pragma hdrstop

#include "random.h"

//---------------------------------------------------------------------------
#pragma package(smart_init)
//---------------------------------------------------------------------------
/************************* RANROTB.CPP ****************** AgF 1999-03-03 *
*  Random Number generator 'RANROT' type B                               *
*                                                                        *
*  This is a lagged-Fibonacci type of random number generator with       *
*  rotation of bits.  The algorithm is:                                  *
*  X[n] = ((X[n-j] rotl r1) + (X[n-k] rotl r2)) modulo 2^b               *
*                                                                        *
*  The last k values of X are stored in a circular buffer named          *
*  randbuffer.                                                           *
*  The code includes a self-test facility which will detect any          *
*  repetition of previous states.                                        *
*  The function uses a fast method for conversion to floating point.     *
*  This method relies on floating point numbers being stored in the      *
*  standard 64-bit IEEE format.                                          *
*                                                                        *
*  The theory of the RANROT type of generators and the reason for the    *
*  self-test are described at www.agner.org/random/ranrot.htm            *
*                                                                        *
*************************************************************************/
//---------------------------------------------------------------------------
// constructor:
TRanrotBGenerator::TRanrotBGenerator(uint32 seed) {
  RandomInit(seed);  SetInterval(0, 99);}

//---------------------------------------------------------------------------
// returns a random number between 0 and 1:
double TRanrotBGenerator::Random() {
  uint32 x;
  // generate next random number
  x = randbuffer[p1] = _lrotl(randbuffer[p2], R1) + _lrotl(randbuffer[p1], R2);
  // rotate list pointers
  if (--p1 < 0) p1 = KK - 1;
  if (--p2 < 0) p2 = KK - 1;
#ifdef SELF_TEST
  // perform self-test
  if (randbuffer[p1] == randbufcopy[0] &&
    memcmp(randbuffer, randbufcopy+KK-p1, KK*sizeof(uint32)) == 0) {
      // self-test failed
      if ((p2 + KK - p1) % KK != JJ) {
        // note: the way of printing error messages depends on system
        // In Windows you may use FatalAppExit
        printf("Random number generator not initialized");}
      else {
        printf("Random number generator returned to initial state");}
      exit(1);}
#endif
  // fast conversion to float:
  union {
    double randp1;
    uint32 randbits[2];};
  randbits[0] = x << 20;
  randbits[1] = (x >> 12) | 0x3FF00000;
  return randp1 - 1.0;}
//---------------------------------------------------------------------------
int TRanrotBGenerator::iRandom() {
  // get integer random number
  int i = iinterval * Random();
  if (i >= iinterval) i = iinterval-1;
  return imin + i;}
//---------------------------------------------------------------------------
// set interval for iRandom
void TRanrotBGenerator::SetInterval(int min, int max) {
  imin = min; iinterval = max - min + 1;}
//---------------------------------------------------------------------------
void TRanrotBGenerator::RandomInit (uint32 seed) {
  // this function initializes the random number generator.
  int i;
  // make sure seed != 0
  if (seed==0) seed--;

  // make random numbers and put them into the buffer
  for (i=0; i<KK; i++) {
    seed ^= seed << 13; seed ^= seed >> 17; seed ^= seed << 5;
    randbuffer[i] = seed;}

  // check that the right data formats are used by compiler:
  union {
    double randp1;
    uint32 randbits[2];};
  randp1 = 1.5; assert(randbits[1]==0x3FF80000);

  // initialize pointers to circular buffer
  p1 = 0;  p2 = JJ;
#ifdef SELF_TEST
  // store state for self-test
  memcpy (randbufcopy, randbuffer, KK*sizeof(uint32));
  memcpy (randbufcopy+KK, randbuffer, KK*sizeof(uint32));
#endif
  // randomize some more
  for (i=0; i<9; i++) Random();
}
//---------------------------------------------------------------------------
